<?php

/**
 * Initializes the BPVMWP plugin.
 *
 * @package BPVMWP
 */
namespace BPVMWP;

/**
 * Class Init
 *
 * Initializes and registers all the services for the plugin.
 *
 * @package BPVMWP
 */
class Init {

	/**
     * Retrieves the list of services to be registered
     *
     * @since 2.0.6
     */
	public static function get_services() {

		/**
		 * Add plugin required classes.
        *
		 * @since 1.0.0
		 */

		$services = [];

		$service_classes = [
			'helpers'           => self::get_helper_classes(),
			'base'              => self::get_base_classes(),
			'meta'              => self::get_meta_classes(),
			'actions'           => self::get_action_classes(),
			'filters'           => self::get_filter_classes(),
			'shortcode'         => self::get_shortcode_classes(),
			'notices'           => self::get_notices_classes(),
			'plugin_pages'      => self::get_plugin_pages_classes(),
			'dashboard_widgets' => self::get_dashboard_widget_classes(),
			'voting_report'     => self::get_voting_report_classes(),
		];

		foreach ( $service_classes as $service_class ) {
			$services = array_merge( $services, $service_class );
		}

		return $services;
	}

	/**
	 * Registered all the classes.
     *
	 * @since 1.0.0
	 */
	public static function register_services() {

		if ( empty( self::get_services() ) ) {
			return;
		}

		foreach ( self::get_services() as $service ) {

			$service = self::instantiate( $service );

			if ( method_exists( $service, 'register' ) ) {
				$service->register();
			}
		}
	}

	/**
	 * Instantiate all the registered service classes.
     *
     * @param   class $service The class to instantiate.
     * @author   Md Mahbub Alam Khan
     * @return  object
     * @since   1.1.0
	 */
	private static function instantiate( $service ) {
		return new $service();
	}

	/**
	 * Get Base classes.
	 *
	 * @return array
	 */
	private static function get_base_classes() {
		$classes = [
			Base\Enqueue::class,
			Base\AdminEnqueue::class,
			Base\Language::class,
			Base\FrontendInlineScripts::class,
			Base\FrontendInlineStyles::class,
			Base\FrontendFilterHandlers::class,
			Base\FrontendAjaxHandlers::class,
			Base\AdminAjaxHandlers::class,
			Base\AdminActionHandlers::class,
			Base\IncludePluginFiles::class,
			Base\ActivationRedirect::class,
			Base\PluginInstallation::class,
			Base\PluginUpdate::class,
		];
		return $classes;
	}

	/**
	 * Get Helper classes.
	 *
	 * @return array
	 */
	private static function get_helper_classes() {
		$classes = [
			Helpers\PluginConstants::class,
			Helpers\SchedulerConstants::class,
			Helpers\PluginColorScheme::class,
			Helpers\BpvmHelper::class,
			Helpers\Common::class,
		];
		return $classes;
	}

	/**
	 * Get Meta classes.
	 *
	 * @return array
	 */
	private static function get_meta_classes() {
		$classes = [
			Controllers\PluginMeta\MetaInfo::class,
		];
		return $classes;
	}
	/**
	 * Get Plugin pages classes.
	 *
	 * @return array
	 */
	private static function get_plugin_pages_classes() {
		$classes = [
			Controllers\Pages\PluginPages::class,
			Controllers\Pages\ReportDownloader::class,

		];
		return $classes;
	}

	/**
	 * Get Filter classes.
	 *
	 * @return array
	 */
	private static function get_action_classes() {

		$classes = [
			Controllers\Actions\Admin\QuickBulkEdit::class,
			Controllers\Actions\Admin\Scheduler::class,
		];
		return $classes;
	}

	/**
	 * Get Filter classes.
	 *
	 * @return array
	 */
	private static function get_filter_classes() {

		$classes = [
			Controllers\Admin\CustomColumns::class,
		];
		return $classes;
	}


	/**
	 * Get Shortcode classes.
	 *
	 * @return array
	 */
	private static function get_shortcode_classes() {
		$classes = [
			Controllers\Shortcodes\VoteLayout::class,
			Controllers\Shortcodes\TaxonomyFilter::class,
			Controllers\Shortcodes\PostsList::class,
			Controllers\Shortcodes\PostsFilter::class,
			Controllers\Shortcodes\ShareBtn::class,
			Controllers\Shortcodes\VoteStats::class,
		];
		return $classes;
	}

	/**
	 * Get Notices classes.
	 *
	 * @return array
	 */
	private static function get_notices_classes() {
		$classes = [
			Controllers\Notices\PluginNotices::class,
			Controllers\Notices\PluginNoticesAjaxHandler::class,
		];
		return $classes;
	}

	/**
	 * Get Dashboard Widget classes.
	 *
	 * @return array
	 */
	private static function get_dashboard_widget_classes() {

		$classes = [
			Controllers\DashboardWidgets\PluginDashboardWidgets::class,
		];
		return $classes;
	}

	/**
	 * Get Voting Report classes.
	 *
	 * @return array
	 */
	private static function get_voting_report_classes() {
		$classes = [
			Controllers\Admin\Report\CalculateAllVotes::class,
		];
		return $classes;
	}
}
