<?php
namespace BPVMWP\Helpers;

/**
 * Class for plugin constants.
 *
 * @package BPVMWP
 */
class PluginConstants {

	/**
     * Static property to hold plugin options.
     *
     * @var array
     */
	public static $plugin_options = [];

    /**
     * Initialize the plugin options.
     */
	public static function init() {

		define( 'BPVM_PLUGIN_OPTIONS_ID', 'bwl_pvm_options' );
		self::$plugin_options = get_option( BPVM_PLUGIN_OPTIONS_ID );

		// Plugin Installation Date Time.
		define( 'BPVM_PRODUCT_INSTALLATION_DATE', 'bpvm_installation_date' );
		if ( empty( get_option( BPVM_PRODUCT_INSTALLATION_DATE ) ) ) {
			update_option( BPVM_PRODUCT_INSTALLATION_DATE, date( 'Y-m-d H:i:s' ) );
		}

	}

	/**
	 * Get the absolute path to the plugin root.
	 *
	 * @return string
	 * @example wp-content/plugins/bwl-pro-voting-manager/
	 */
	public static function get_plugin_path(): string {
		return dirname( dirname( __DIR__ ) ) . '/';
	}

	/**
	 * Get the plugin URL.
	 *
	 * @return string
	 * @example http://BPVMwp.local/wp-content/plugins/bwl-pro-voting-manager/
	 */
	public static function get_plugin_url(): string {
		return plugin_dir_url( self::get_plugin_path() . BPVM_PLUGIN_ROOT_FILE );
	}

	/**
	 * Register the plugin constants.
	 */
	public static function register() {
		self::init();
		self::set_paths_constants();
		self::set_base_constants();
		self::set_icon_constants();
		self::set_nonce_constants();
		self::set_assets_constants();
		self::set_custom_tables();
		self::set_meta_data_constants();
		self::set_updater_constants();
		self::set_admin_pages_constants();
		self::set_product_info_constants();
	}

	/**
	 * Set the plugin base constants.
	 */
	public static function set_base_constants() {
		// This is super important to check if the get_plugin_data function is already loaded or not.
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data = get_plugin_data( BPVM_PLUGIN_DIR . BPVM_PLUGIN_ROOT_FILE );

		define( 'BPVM_PLUGIN_VERSION', $plugin_data['Version'] ?? '1.0.0' );
		define( 'BPVM_PLUGIN_TITLE', $plugin_data['Name'] ?? 'BWL Pro Voting Manager' );
		define( 'BPVM_TRANSLATION_DIR', $plugin_data['DomainPath'] ?? '/lang/' );
		define( 'BPVM_TEXT_DOMAIN', $plugin_data['TextDomain'] ?? '' );

		define( 'BPVM_PLUGIN_FOLDER', 'bwl-pro-voting-manager' );
		define( 'BPVM_CURRENT_VERSION', BPVM_PLUGIN_VERSION );
	}

	/**
	 * Set the plugin icon constants.
	 */
	public static function set_icon_constants() {
		define( 'BPVM_ICON_ERROR', '❌' );
		define( 'BPVM_ICON_SUCCESS', '✅' );
		define( 'BPVM_ICON_WARNING', '⚠️' );
	}

	/**
	 * Set the plugin nonce constants.
	 */
	public static function set_nonce_constants() {
		define( 'BPVM_FRONTEND_NONCE_ID', 'bpvm_frontend_nonce' );
		define( 'BPVM_ADMIN_NONCE_ID', 'bpvm_admin_nonce' );
	}

	/**
	 * Set the plugin paths constants.
	 */
	private static function set_paths_constants() {
		define( 'BPVM_PLUGIN_ROOT_FILE', 'bwl-pro-voting-manager.php' );
		define( 'BPVM_PLUGIN_DIR', self::get_plugin_path() );
		define( 'BPVM_PLUGIN_FILE_PATH', BPVM_PLUGIN_DIR );
		define( 'BPVM_PLUGIN_URL', self::get_plugin_url() );

		define( 'BPVM_CONTROLLER_DIR', BPVM_PLUGIN_DIR . 'includes/Controllers/' );
		define( 'BPVM_VIEWS_DIR', BPVM_PLUGIN_DIR . 'includes/Views/' );
	}

	/**
	 * Set the plugin assets constants.
	 */
	private static function set_assets_constants() {
		define( 'BPVM_STYLES_ASSETS_DIR', BPVM_PLUGIN_URL . 'assets/styles/' );
		define( 'BPVM_SCRIPTS_ASSETS_DIR', BPVM_PLUGIN_URL . 'assets/scripts/' );
		define( 'BPVM_PLUGIN_LIBS_DIR', BPVM_PLUGIN_URL . 'libs/' );
	}

	/**
	 * Set the custom table names.
	 */
	private static function set_custom_tables() {
		global $wpdb;
		define( 'TABLE_BPVM_DATA', $wpdb->prefix . 'bpvm_data' );
		define( 'TABLE_BPVM_SUMMARY', $wpdb->prefix . 'bpvm_summary' );
	}

	/**
	 * Set the Meta data constants.
	 */
	private static function set_meta_data_constants() {
		define( 'BPVM_VOTE_BOX_STATUS', 'bwl_pvm_display_status' );
		define( 'BPVM_TOTAL_VOTES', 'pvm_total_votes' );

		define( 'BPVM_LIKE_VOTES_COUNT', 'pvm_like_votes_count' );
		define( 'BPVM_LIKED_VOTES_BY_DATE', 'pvm_liked_votes_by_date' );

		define( 'BPVM_DISLIKE_VOTES_COUNT', 'pvm_dislike_votes_count' );
		define( 'BPVM_DISLIKED_VOTES_BY_DATE', 'pvm_disliked_votes_by_date' );

		define( 'BPVM_VOTED_IP', 'pvm_voted_ip' );
		define( 'BPVM_VOTE_DATE', 'vote_date' );
		define( 'BPVM_VOTE_START_DATE', 'bpvm_vote_start_date' );
		define( 'BPVM_VOTE_END_DATE', 'bpvm_vote_end_date' );

		// Added version 1.4.7 (Use for chart.js and summary report)
		// Action Scheduler constants.
		define( 'BPVM_DATE_WISE_VOTES_COUNT', 'bpvm_date_wise_votes_count' );
		define( 'BPVM_ALL_POSTS_LIKES_COUNT', 'bpvm_all_posts_likes_count' );
		define( 'BPVM_ALL_POSTS_DISLIKES_COUNT', 'bpvm_all_posts_dislikes_count' );
	}

	/**
	 * Set the updater constants.
	 */
	private static function set_updater_constants() {
		define( 'BPVM_UPDATER_SLUG', BPVM_PLUGIN_FOLDER . '/' . BPVM_PLUGIN_ROOT_FILE ); // phpcs:ignore
		define( 'BPVM_PATH', BPVM_PLUGIN_DIR );
	}

	/**
	 * Set the admin pages constants.
	 */
	private static function set_admin_pages_constants() {
		define( 'BPVM_ADMIN_PARENT_MENU_SLUG', 'bwl-pvm' );
		define( 'BPVM_PRODUCT_LICENSE_PAGE', 'admin.php?page=bpvm-license' );
		define( 'BPVM_PRODUCT_LICENSE_PAGE_OFFLINE', BPVM_PRODUCT_LICENSE_PAGE . '&bwlverify=offline' );
	}

	/**
	 * Set the product info constants.
	 */
	private static function set_product_info_constants() {
		define( 'BPVM_PRODUCT_ID', '7616885' );
		define( 'BPVM_PRODUCT_INSTALLATION_TAG', 'bpvm_installation_' . str_replace( '.', '_', BPVM_PLUGIN_VERSION ) );
		define( 'BPVM_PRODUCT_URL', 'https://1.envato.market/bpvm-wp' );
		define( 'BPVM_PURCHASE_VERIFIED_KEY', 'bpvm_purchase_verified' );
		define( 'BPVM_PURCHASE_INFO_KEY', 'bpvm_purchase_info' );
		define( 'BPVM_PRODUCT_SUPPORT', 'https://codecanyon.net/item/bwl-pro-voting-manager/7616885/support/contact' );
		define( 'BPVM_PRODUCT_YOUTUBE_PLAYLIST', 'https://www.youtube.com/playlist?list=PLxYTuQlgnCLqNxc2N7XUDYllIJjJB_-tU' );
		define( 'BPVM_PRODUCT_DOC', 'https://projects.bluewindlab.net/wpplugin/bpvm/doc' );
		define( 'BPVM_PRODUCT_AUTHOR_PROFILE', 'https://codecanyon.com/user/xenioushk' );

		define( 'BPVM_ADMIN_OPTIONS_PANEL_URL', admin_url( 'admin.php?page=bwl-pvm_option_panel' ) );
		define( 'BPVM_ADMIN_REPORT_PANEL_URL', admin_url( 'admin.php?page=bpvm-voting-report' ) );
		define( 'BPVM_ADMIN_ADDONS_URL', admin_url( 'admin.php?page=bpvm-addons' ) );
	}
}
