<?php
namespace BPVMWP\Helpers;

/**
 * Class for Logged In Vote Status Checker.
 *
 * @package BPVMWP
 */
class LoggedInStatusChecker {

    /**
     * Check if the user has already voted once.
     *
     * @param  mixed  $bpvm_share_btn Share button HTML.
     * @param  int    $postId        Post ID.
     * @param  string $pvm_msg_thanks Thank you message.
     * status=1 (Already voted), status=0 (Not voted yet)
     * pvm_clear_counter=1 (remove the old vote counter), pvm_clear_counter=0 (keep the old counter)
     * @return array ['status', 'pvm_clear_counter', 'msg']
     */
	public static function check_logged_in_user_vote_status( $bpvm_share_btn, $postId, $pvm_msg_thanks = '' ) {

		// Get all the logged in voted information.
		// ID PATTERN: 'pvm_logged_voted_info_user_id_post_id'

		$userId = 0;

		if ( is_user_logged_in() ) :
			$current_user = wp_get_current_user();
			$userId       = $current_user->ID;
		endif;

		// Get logged in users vote data
		$loggedInVoteData = get_option( 'pvm_logged_voted_info' );

		// Unique key for each post vote.
		// @Pattern: pvm_logged_<userId>_<postId>

		$loggedInVoteId = 'pvm_logged_' . $userId . '_' . $postId;

		// Logged in users can not submit more than 1 vote.
		// Time count.

		$pvm_vote_submission_status = isset( $loggedInVoteData[ $userId ] ) && in_array( $loggedInVoteId, array_keys( $loggedInVoteData[ $userId ] ) ) ? 1 : 0;

		$bwl_pvm_options = get_option( 'bwl_pvm_options' );

		$pvm_max_vote_submission = ( isset( $bwl_pvm_options['pvm_max_vote_submission'] ) && is_numeric( $bwl_pvm_options['pvm_max_vote_submission'] ) ) ? $bwl_pvm_options['pvm_max_vote_submission'] : 1; // = 1 votes.

		$timebeforerevote = ( isset( $bwl_pvm_options['pvm_vote_interval'] ) && is_numeric( $bwl_pvm_options['pvm_vote_interval'] ) ) ? $bwl_pvm_options['pvm_vote_interval'] : 120; // Default Interval = 2 hours

		$loggedInUserVoteData = self::get_user_post_vote_info( $postId, $userId );

		// Check Submission Count.

		if ( $loggedInUserVoteData['total_votes'] < $pvm_max_vote_submission ) {
			$pvm_vote_submission_status = false;
		}

		// Check Date Validity.

		$now  = time();
		$time = strtotime( $loggedInUserVoteData['last_vote_date'] );

		if ( round( ( $now - $time ) / 60 ) > $timebeforerevote ) {
			$pvm_vote_submission_status = false;
		}

		if ( $pvm_vote_submission_status ) {

			// echo "Information EXIST in Database";

			$pvm_msg_already_voted = apply_filters( 'pvm_msg_vote_exist', esc_html__( 'You have already submitted your vote!', 'bwl-pro-voting-manager' ) );

			$pvm_vote_checking_result = [
				'status'            => 1,
				'pvm_clear_counter' => 0,
				'msg'               => $pvm_msg_already_voted,
			];
		} else {

			// echo "NOT EXIST";
			$loggedInVoteData[ $userId ][ $loggedInVoteId ] = 'true';
			update_option( 'pvm_logged_voted_info', $loggedInVoteData );

			$pvm_vote_checking_result = [
				'status'            => 0, // You can submit vote,
				'pvm_clear_counter' => 1,
				'msg'               => $pvm_msg_thanks . $bpvm_share_btn,
			];
		}

		return $pvm_vote_checking_result;
	}


    /**
     *
     * @param int $postId
     * @param int $userId
     *
     * @return array ['total_votes'=> 10, 'last_vote_date'=> '2023-10-21']
     */
	public static function get_user_post_vote_info( $postId = 0, $userId = 0 ) {

		global $wpdb;
		$bpvm_voting_data_table = TABLE_BPVM_DATA; // for deatils. each day info.

		$vars = [ $postId, $userId, date( 'Y-m-d' ) ];

		$con_mv_post_filters = '';

		$sql = $wpdb->prepare(
            "SELECT ID, vote_date FROM {$bpvm_voting_data_table} 
                    WHERE postid = %d 
                    AND user_id = %d 
                    AND DATE(vote_date) = %s 
                    " . $con_mv_post_filters . '
                    ORDER BY vote_date DESC', $vars
		);

		$result = $wpdb->get_results( $sql, ARRAY_A );

		$userVoteData = [];

		if ( ! empty( $result ) ) {

			$userVoteData['total_votes']    = count( $result );
			$userVoteData['last_vote_date'] = $result[0]['vote_date'];
		} else {
			$userVoteData['total_votes']    = 0;
			$userVoteData['last_vote_date'] = '';
		}

		return $userVoteData;
	}
}
