<?php
namespace BPVMWP\Helpers;

/**
 * Class for Category Wise Vote Status Checker.
 *
 * @package BPVMWP
 */
class CategoryWiseVoteStatusChecker {

    /**
     * Get the parent category ID of a post.
     *
     * @param  int $postId Post ID.
     * @return array Parent category ID and name.
     */
    public static function get_parent_category_id( $postId = '' ) {

        $categories = get_the_category( $postId );

        // Initialize a variable to store the first category
        $catInfo = [];

        // Check if there are categories assigned to the post
        if ( $categories ) {
            // Retrieve the first category from the categories array
            $catInfo = $categories[0];
        }

        // Output the first category name and link
        if ( ! empty( $catInfo ) ) {
            // echo '<a href="' . esc_url(get_category_link($first_category->term_id)) . '">' . esc_html($first_category->name) . '</a>';

            return [
                'catId'   => $catInfo->term_id,
                'catName' => $catInfo->term_id,
            ];
        } else {
            return [];
        }
    }


    /**
     * Restrict voting based on category.
     *
     * @param  mixed  $bpvm_share_btn Share button HTML.
     * @param  int    $postId        Post ID.
     * @param  string $pvm_msg_thanks Thank you message.
     * status=1 (Already voted), status=0 (Not voted yet)
     * pvm_clear_counter=1 (remove the old vote counter), pvm_clear_counter=0 (keep the old counter)
     * @return array ['status', 'pvm_clear_counter', 'msg']
     */
	public static function restrict_category_wise_voting( $bpvm_share_btn, $postId, $pvm_msg_thanks = '' ) {

		// Get all the logged in voted information.
		// ID PATTERN: 'pvm_cwvd_<user_id>_<cat_id>_<post_id>'

		$voteSubmitStatusData = [];
		$userId               = 0;
		$catInfo              = self::get_parent_category_id( $postId ); // general topic. we will change that later.

		// Check the category status.

		if ( empty( $catInfo ) ) {
			$voteSubmitStatusData = [
				'status'            => 1,
				'pvm_clear_counter' => 0,
				'msg'               => esc_html__( 'This post must have a category for voting!', 'bwl-pro-voting-manager' ),
			];
		}

		if ( is_user_logged_in() ) :
			$current_user = wp_get_current_user();
			$userId       = $current_user->ID;
		endif;

		// Get logged in users vote data
		$categoryWiseVoteData = get_option( 'pvm_category_wise_vote_data' );

		// Unique key for each post vote.
		// @Pattern: pvm_cwvd_<user_id>_<cat_id>

		$categoryWiseVoteId = 'pvm_cwvd_' . $userId . '_' . $catInfo['catId']; // fixed in version 1.3.9

		// Logged in users can not submit more than 1 vote.
		// Time count.

		$voteSubmitStatus = isset( $categoryWiseVoteData[ $userId ] ) && in_array( $categoryWiseVoteId, array_keys( $categoryWiseVoteData[ $userId ] ) ) ? 1 : 0;

		$pvm_max_vote_submission = 1; // = 1 votes.

		if ( $voteSubmitStatus ) {

			$voteSubmitStatusData = [
				'status'            => 1,
				'pvm_clear_counter' => 0,
				'msg'               => apply_filters( 'pvm_msg_cat_vote_exist', esc_html__( 'You are allowed to submit one vote for each category!', 'bwl-pro-voting-manager' ) ),
			];
		} else {

			// echo "NOT EXIST";
			$categoryWiseVoteData[ $userId ][ $categoryWiseVoteId ] = 'true';
			update_option( 'pvm_category_wise_vote_data', $categoryWiseVoteData );

			$voteSubmitStatusData = [
				'status'            => 0, // You can submit vote,
				'pvm_clear_counter' => 1,
				'msg'               => $pvm_msg_thanks . $bpvm_share_btn,
			];
		}

		return $voteSubmitStatusData;
	}
}
