<?php
namespace BPVMWP\Helpers;

use DateTime;

/**
 * Class for BPVM helper functions
 *
 * @package BPVMWP
 */
class BpvmHelper {


    /**
     * Beautify the date.
     *
     * @param string $date The date to beautify.
     *
     * @return string
     */
    public static function beautify_date( $date ) {

		$splitDate = explode( 'T', $date );

		$rearrangeDate = explode( '-', $splitDate[0] );

		return $rearrangeDate[2] . '-' . $rearrangeDate[1] . '-' . $rearrangeDate[0];
	}

    /**
     * Get the days left for renewal.
     *
     * @param string $renewalDate The renewal date.
     *
     * @return array
     */
    public static function get_renewal_days_left( $renewalDate = '' ) {

		if ( empty( $renewalDate ) ) {
			return [
				'status'    => 0,
				'days_left' => 0,
				'msg'       => 'Renewal date is not set',
			];
		}

		// Convert the renewal date to a DateTime object
		$renewalDateTime = new DateTime( $renewalDate );

		// Get the current date
		$currentDate = new DateTime();

		// Compare the renewal date with the current date
		if ( $renewalDateTime < $currentDate ) {
			// Renewal date has expired

			return [
				'status'    => 0,
				'days_left' => 0,
				'msg'       => 'Renewal date has expired',
			];
		} else {
			// Calculate the difference between the renewal date and the current date
			$interval = $currentDate->diff( $renewalDateTime );
			$daysLeft = $interval->days;

			return [
				'status'    => 1,
				'days_left' => $daysLeft,
				'msg'       => "$daysLeft days left for renewal",
			];
		}
	}

	/**
	 * Get the vote counts.
	 * This function returns total vote counts number for like and dislike.
     *
	 * @param int $post_id The post ID.
	 * @param int $vote_type The vote type (1 for like, 2 for dislike).
	 * @example 1=like, 2=dislike.
	 * @return int
	 */
	public static function bpvm_get_vote_counts_meta( $post_id = 0, $vote_type = 1 ) {

		$votesCount = ( $vote_type === 2 ) ?
											get_post_meta( $post_id, BPVM_DISLIKE_VOTES_COUNT, true ) :
											get_post_meta( $post_id, BPVM_LIKE_VOTES_COUNT, true );

		return ( empty( $votesCount ) ) ? 0 : intval( $votesCount );
	}

	/**
	 * Get the vote box display status.
     *
	 * @param int $post_id The post ID.
	 * @return int
	 */
	public static function get_vote_box_display_status( $post_id = 0 ) {

		if ( $post_id === 0 ) {
			return 0;
		}

		return intval( get_post_meta( $post_id, BPVM_VOTE_BOX_STATUS, true ) );

	}

	/**
	 * Calculate the percentage.
	 *
	 * @param int $num_total The total number.
	 * @param int $num_amount The amount to calculate the percentage for.
	 * @return float
	 */
	public static function calculate_percentage( $num_total = 0, $num_amount = 0 ) {

		if ( $num_amount == 0 ) {

			return 0;
		}

		$count1 = $num_amount / $num_total;
		$count2 = $count1 * 100;
		$count  = number_format( $count2, 4 );
		return $count;
	}

	/**
	 * Check if the user has crossed the vote limit.
	 *
	 * @param int $post_id The post ID.
	 * @return bool
	 */
	public static function check_vote_limit_cross_status( $post_id = 0 ) {

		$options = PluginConstants::$plugin_options;

		$max_votes = ( isset( $options['pvm_max_vote_submission'] ) && is_numeric( $options['pvm_max_vote_submission'] ) ) ? $options['pvm_max_vote_submission'] : 1; // = 1 votes.

		// Get no of submissions from a single IP address.
		$submission_counter = get_post_meta( $post_id, 'pvm_submission_counter' );

		$user_ip = $_SERVER['REMOTE_ADDR'];

		if ( ! empty( $submission_counter ) && is_array( $submission_counter[0] ) &&
					in_array( $user_ip, array_keys( $submission_counter[0] ) ) ) {
			// store meta submission counter value.
			$total_submitted = $submission_counter[0][ $user_ip ];

		} else {

			// if no one submitted voted yet, initialize pvm_submission_counter variable.
			$total_submitted = 0;

		}

		$limit_cross_status = false;

		if ( $total_submitted >= $max_votes ) {

			$limit_cross_status = true;
		}

		return $limit_cross_status;
	}
}
