<?php
namespace BPVMWP\Controllers\Admin\Report;

use BPVMWP\Helpers\BpvmHelper;

use Xenioushk\BwlPluginApi\Api\Database\QueryManagerApi;

/**
 * Custom Columns
 *
 * @package BPVMWP
 */
class CalculateAllVotes {

	/**
	 * Summary table query manager instance.
	 *
	 * @var object
	 */
	private $summary_table_query;

	/**
	 * All date wise votes info
     *
	 * @var object
	 */
	private $all_votes_info = [];

	/**
	 * Register the column filters.
	 */
	public function register() {

		$this->summary_table_query = new QueryManagerApi( TABLE_BPVM_SUMMARY );
		$this->count_all_posts_vote_data();

	}

	/**
	 * Get summary table data.
	 *
	 * @return array
	 */
	private function get_summary_table_data() {

		$selected_fields = 'SUM(total_votes) AS total_votes,vote_type,vote_date';

		$order_by  = 'vote_date'; // Order by vote date.
		$order_dir = 'DESC'; // Order by ascending order.
		$group_by  = [ 'vote_type', 'vote_date' ]; // Group by vote type and vote date.

		$args = [
			'selected_fields' => $selected_fields, // default: *. Example: 'id,name,email,age'
			'order_by'        => $order_by, // Or, any of your table column
			'order_dir'       => $order_dir, // ASC/DESC
			'group_by'        => $group_by,
		];

		return $this->summary_table_query->get_items( $args );
	}

	/**
	 * Display Custom Columns Header For Voting Manager Plugin.
	 *
	 * @param bool $fetch_fresh_data Whether to fetch fresh data or not.
	 * @return array
	 * @since 1.4.7
	 */
	public function fetch_all_votes_info( $fetch_fresh_data = false ) {
		if ( ! $fetch_fresh_data ) {
			// 86400 = 24 hours in seconds.
			$interval = 86400; // 24 hours in seconds.

			$now               = date( 'Y-m-d H:i:s' );
			$last_updated_data = get_option( BPVM_DATE_WISE_VOTES_COUNT )['last_updated'] ?? '';

			if ( strtotime( $last_updated_data ) > strtotime( $now ) - $interval ) {
				// If the last updated time is within the last 24 hours, return the stored data.
				return $this->fetch_stored_data();
			}
		}

		$results = $this->get_summary_table_data();
		$data    = ( ! empty( $results['data'] ) ) ? $results['data'] : [];

		$final_data = [
			// we will compare that will current date time. Generally update every 24 hours.
			'last_updated' => date( 'Y-m-d H:i:s' ),
			'data'         => $data,
			'fresh'        => true,
		];

		update_option( BPVM_DATE_WISE_VOTES_COUNT, $final_data );
		return $final_data;

	}

	/**
	 * Fetch stored data from the database.
	 *
	 * @return array
	 */
	private function fetch_stored_data() {
		return get_option( BPVM_DATE_WISE_VOTES_COUNT, [] );
	}

	/**
	 * Count all posts vote data both likes and dislikes.
	 * We will call this function from action scheduler.
	 */
	public function count_all_posts_vote_data() {

		$this->all_votes_info = $this->fetch_all_votes_info();
		$fresh_data           = $this->all_votes_info['fresh'] ?? false;

		if ( $fresh_data ) {
			$this->set_total_likes_dislikes_count();
		}
	}

	/**
	 *
	 * 🚨  DO NOT DELETE THIS 🚨
     * Use this function to Refresh all posts vote data both likes and dislikes.
     * This function will be used for refreshing the votes data from admin dashboard and action scheduler.
     *
     * @param QueryManagerApi $summary_table_query Summary table query manager instance.
	 * @return array
	 */
	public function refresh_votes_data( $summary_table_query ) {

		$this->summary_table_query = $summary_table_query;
		$this->all_votes_info      = $this->fetch_all_votes_info( true );
		$this->set_total_likes_dislikes_count();
		return self::get_total_votes_counts();
	}

	/**
	 * Set total likes and dislikes count.
	 */
	private function set_total_likes_dislikes_count() {

		$total_likes    = 0;
		$total_dislikes = 0;

		if ( ! empty( $this->all_votes_info['data'] ) ) {

			foreach ( $this->all_votes_info['data'] as $vote_data ) {
				if ( intval( $vote_data['vote_type'] ) === 1 ) { // like
					$total_likes += intval( $vote_data['total_votes'] );
				} elseif ( intval( $vote_data['vote_type'] ) === 2 ) { // dislike
					$total_dislikes += intval( $vote_data['total_votes'] );
				}
			}
		}

		update_option( BPVM_ALL_POSTS_LIKES_COUNT, $total_likes );
		update_option( BPVM_ALL_POSTS_DISLIKES_COUNT, $total_dislikes );
	}

	/**
	 * Get total votes counts.
	 *
	 * @param string $type Type of vote count to retrieve. likes, dislikes, or both.
	 *                     1 for likes, 2 for dislikes, empty for both.
	 * @return int|array Total votes count or array of likes and dislikes counts.
	 */
	public static function get_total_votes_counts( $type = '' ) {

		switch ( $type ) {
			case 1: // Likes
				return get_option( BPVM_ALL_POSTS_LIKES_COUNT, 0 );
			case 2: // Dislikes
				return get_option( BPVM_ALL_POSTS_DISLIKES_COUNT, 0 );
			default:
				return [
					'likes'    => get_option( BPVM_ALL_POSTS_LIKES_COUNT, 0 ),
					'dislikes' => get_option( BPVM_ALL_POSTS_DISLIKES_COUNT, 0 ),
				];
		}

	}
}
